/////////////////////////////////////////////////////////////////////////////// 
// 
// Pager.cc 
// ---------- 
// Curses based pager implementation 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2017 Bjoern Lemke 
// 
// IMPLEMENTATION MODULE 
// 
// Class: Monitor 
// 
// Description: Cursor based pager implementation 
// 
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 

#include <stdio.h> 
#include <stdlib.h> 
#include <unistd.h> 

#include "Pager.h" 
#include "Tokenizer.h"
#include "Sleeper.h" 

#define COL_HEADER 1 
#define COL_FOOTER 2 
#define COL_CONTENT 4 
#define COL_SELECTED 5

#define LEFTOFFSET 1 
#define TOPOFFSET 1 

#define KEY_TAB 9 
#define KEY_RETURN 10 
#define KEY_ESC 27 
#define KEY_DEL 127 

#define CUR_ROW_OFFSET 3 
#define CUR_COL_OFFSET 20 
#define VAL_MAX_LEN 30 
#define VAL_MAX_NUM 20 
#define OK_COL_POS 6 
#define ABORT_COL_POS 16

#define PAGESIZE 10

#define MSEC_SLEEPDELAY 1000
#define MSEC_RESIZEDELAY 50

Pager::Pager() : SigHandler() 
{
    
#ifndef LFCNOCURSES

/*  Initialize ncurses  */ 
    initscr(); 
    curs_set(0); 
    start_color(); 
   
    
    init_pair(COL_HEADER,  COLOR_WHITE,     COLOR_BLUE); 
    init_pair(COL_FOOTER,  COLOR_WHITE,     COLOR_BLUE); 
    init_pair(COL_CONTENT,  COLOR_WHITE,     COLOR_BLACK); 
    init_pair(COL_SELECTED,  COLOR_GREEN,     COLOR_BLACK);
    
    noecho();
    keypad(stdscr, TRUE); 
    timeout(0); 

    wbkgd(stdscr, COLOR_PAIR(COL_CONTENT));
    
    init(); 

#ifndef HAVE_MINGW 
    install(SIGWINCH);
#endif

 
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

} 

Pager::~Pager() 
{ 
#ifndef LFCNOCURSES 
// we need endwin and erase to reset to original terminal settings
    endwin();
    erase();
#endif 
} 


void Pager::sigCatch(int sig) 
{ 

    try 
    { 
#ifndef LFCNOCURSES 
        // clear(); 
        endwin(); 

	_rows = LINES;
	_cols = COLS;
        initscr();
        clear(); 
        refresh(); 
#endif

	Sleeper::milliSleep(MSEC_RESIZEDELAY);
	
#ifndef HAVE_MINGW 
        install(SIGWINCH); 
#endif

	
    } 
    catch ( Exception e) 
    { 
        Chain msg; 
        e.pop(msg); 
        cout << msg << endl; 
    } 
}; 

void Pager::managePage(const Chain& title, const ListT<Chain>& schema, const ListT< ListT<Chain> >& outData ) 
{
    _title = title;
    _schema = schema;
    formatData(outData);
    
#ifndef LFCNOCURSES 
        
    _rows = LINES;
    _cols = COLS;
	
    int i, ch = 0; 

    _curStat=0;
    _rowSelected=0;
    _vStart=0;
    _rowIdx=0;
    _offset=0;

    do 
    { 	
        switch (ch) 
        { 
	case KEY_DOWN :
	{
	    nextRow();
	    break;	    
	}
	case KEY_UP :
	{
	    prevRow();
	    break;
	}
	case 's' :
	{
	    firstRow();
	    break;
	}
	case 'e' :
	{
	    lastRow();
	    break;
	}
	case 'n' : 
	case ' ' :
	case KEY_NPAGE :
	{
	    nextRow();
	    int pdi=_pdi;
	    int pci=_pci;
	    for ( int i=0; i < PAGESIZE  - 1; i++ )
		nextRow();
	    _pdi=pdi;
	    _pci=pci;
	    break;
	}
	case 'p' :
	case KEY_PPAGE :
	{
	    prevRow();	    
	    int pdi=_pdi;
	    int pci=_pci;

	    for ( int i=0; i < PAGESIZE - 1; i++ )
		prevRow();
	    _pdi=pdi;
	    _pci=pci;
	    
	    break;
	}
	case KEY_RESIZE:
	{
	    _rows = LINES;
	    _cols = COLS;
	    break;  
	}
	default:
	{
	    break;
	}	
	} 

	writePage();
		
	wtimeout(stdscr, MSEC_SLEEPDELAY); 
	ch = wgetch(stdscr);
	
    } 
    while( ch != 'q' );             // continue until user hits q 
        
    // refresh(); 
    // clear();
    // endwin();

#else 
    cout << "Curses not enabled" << endl; 
#endif 
    return; 
}

void Pager::nextRow()
{
    
    if ( _rowIdx < _rowHeight.Size()  - 1 )
    {
	if ( _rowSelected >= _rows + _vStart - _rowHeight[_rowIdx] - 2)
	    _vStart += _rowHeight[_rowIdx];
	else
	    _curStat=1;
	
	_rowSelected += _rowHeight[_rowIdx];

	_pdi = _rowSelected - _rowHeight[_rowIdx];
	_pci = _rowSelected - _rowHeight[_rowIdx] - _vStart + 1;
    
	_ndi = _rowSelected;
	_nci = _rowSelected - _vStart + 1;

	_rowIdx++;
	
    }
    else
    {
	// if end of outData not reached, scroll forward linewise and remember offset
	if ( _vStart < _outData.Size() - _rows + 2)
	{
	    _vStart += 1;
	    _offset++;
	}
    }
}

void Pager::prevRow()
{

    if ( _rowIdx > 0 )
    {
	_rowIdx--;
	
	if (_vStart >= _rowSelected - _rowHeight[_rowIdx] && _vStart > 0)
	{
	    _rowSelected -= _rowHeight[_rowIdx];
	    _vStart = _rowSelected;
	    
	    // _debug = Chain("page mode up, ") + Chain(" RowSelectd=") + Chain(_rowSelected) + Chain(" vstart=") + Chain(_vStart);
	}
	else
	{
	    _curStat=1;
	    _rowSelected -= _rowHeight[_rowIdx];
	    // _debug = Chain("cursor mode up, ") + Chain(" RowSelectd=") + Chain(_rowSelected) + Chain(" vstart=") + Chain(_vStart);

	    _pdi = _rowSelected + _rowHeight[_rowIdx];
	    _pci = _rowSelected-_vStart + _rowHeight[_rowIdx] + 1;
	    
	    _ndi = _rowSelected;
	    _nci = _rowSelected - _vStart + 1;
	}
    }
}

void Pager::firstRow()
{    
    _rowIdx = 0;    
    _vStart = 0;
    _rowSelected = 0;
}

void Pager::lastRow()
{    
    _rowIdx = _rowHeight.Size() - 1;    
    _vStart = _outData.Size() - _rows + 2;
    _rowSelected = _outData.Size() - _rowHeight[_rowIdx];
}

void Pager::writeHeader() 
{ 
#ifndef LFCNOCURSES 

    attron(A_BOLD); 
    wcolor_set(stdscr, COL_HEADER, NULL);

    // clear line
    for(int i = 0; i < _cols; i++) 
    {
	mvprintw(0, i, "%s", (char*)" "); 
    } 

    move(0,0); 

    int col=0;
    Chain *pAttr = _schema.First();
    int colPos=0;
    while ( pAttr ) 
    {
	Chain attr;
	int len;
        getAttrSpec(colPos, *pAttr, attr, len); 
        mvprintw(0, col, "%s", (char*)attr); 

	colPos++;
        col += len;
	pAttr = _schema.Next();
    } 

    attroff(A_BOLD); 
#else 
    throw Exception(EXLOC, "Curses not supported"); 
#endif 

} 

void Pager::writePage()
{

    unsigned int headerSize = 1; 
    unsigned int footerSize = 1; 

    if ( _curStat == 1 )
    {
	// refreshRow(_rowSelected + _rowHeight[_rowIdx], _rowSelected-_vStart + _rowHeight[_rowIdx] + 1);
	// refreshRow(_rowSelected, _rowSelected - _vStart + 1);

	refreshRow(_pdi, _pci);
	refreshRow(_ndi, _nci);
	
	_curStat=0;
	return;
    }
    /*
    else if ( _curStat == 2 )
    {
	refreshRow(_rowSelected - _rowHeight[_rowIdx-1], _rowSelected - _rowHeight[_rowIdx-1] - _vStart + 1);
	refreshRow(_rowSelected, _rowSelected - _vStart + 1);
	_curStat=0;
	return;
    }
    */

    writeHeader(); 

   
    wcolor_set(stdscr, COL_CONTENT, NULL);
    
    for(int i = 0; (i< _rows - headerSize-footerSize) && (i + _vStart < _outData.Size()); i++) 
    { 
	
	for(int j = 0; j < _cols; j++) 
	{
	    mvprintw(i+headerSize, j, "%s", (char*)" "); 
	} 

	int col=0;
	Chain *pAttr = _schema.First();
	int colPos=0;
	while ( pAttr ) 
	{
	    Chain attr;
	    int len;
	    getAttrSpec(colPos, *pAttr, attr, len); 

	    char* pVal = (char*)(_outData[i+_vStart][colPos]);
	    if ( pVal )
		mvprintw(i+headerSize, col, "%s", pVal); 
	    col += len;
	    colPos++;
	    pAttr = _schema.Next();
	} 	    
    }

    refreshRow(_rowSelected, _rowSelected-_vStart+1);

    // writeDebug(); 
    writeFooter(); 
    refresh(); 
}


void Pager::writeFooter()
{
#ifndef LFCNOCURSES

    wcolor_set(stdscr, COL_FOOTER, NULL);

    attron(A_BOLD);
    move(_rows-1,0);
    for(int i = 0; i < _cols; i++)
    {
        mvprintw(_rows-1, i, "%s", (char*)" ");
    }
    mvprintw(_rows-1,0, "up,down,n,p,s,e,q ?");
    mvprintw(_rows-1,21, "%s", (char*)_title);
    mvprintw(_rows-1,_cols-23,"Row %04d ( %04d total)", _rowIdx+1, _rowHeight.Size());
    attroff(A_REVERSE);
#endif
}

void Pager::writeDebug() 
{ 
#ifndef LFCNOCURSES 

    wcolor_set(stdscr, COL_FOOTER, NULL);
    
    attron(A_REVERSE);
    move(_rows-1,0); 
    for(int i = 0; i < _cols; i++) 
    { 
        addch(' '); 
    } 

    mvprintw(_rows-1,0, "%s", (char*)_debug); 

    attroff(A_REVERSE); 
#endif 
} 


void Pager::refreshRow(int dataRow, int pageRow)
{
    int colno=0;
    
    // int rowColor = COL_CONTENT;

    wcolor_set(stdscr, COL_CONTENT, NULL);

    if ( _rowSelected == dataRow )
    {
	wcolor_set(stdscr, COL_SELECTED, NULL);
	wattron(stdscr, A_REVERSE);
	// rowColor = COL_SELECTED;
    }
    
    for(int i = 0; i < _cols; i++) 
    { 
        mvwprintw(stdscr, pageRow, i, "%s", (char*)" ");
    } 

    Chain *pAttr = _schema.First();
    int colPos=0;
    Chain *pVal = _outData[dataRow].First();
    while ( pAttr && pVal )
    {		       
	Chain attr;
	int len;
	getAttrSpec(colPos, *pAttr, attr, len);
		    
	if ( 1 ) // perhaps later we also make right align : lenInfo[0] == 'l' )
	{
	    if ( *pVal )
		mvwprintw(stdscr, pageRow, colno, "%s", (char*)*pVal);
	}
	else
	{
	    int offset = len - pVal->length();
	    if ( *pVal )
		mvwprintw(stdscr, pageRow, colno+offset, "%s", (char*)*pVal);	    
	}

	colPos++;
	colno = colno + len;
	
	// wcolor_set(stdscr, rowColor, NULL);
	
	pVal = _outData[dataRow].Next();
	
	pAttr = _schema.Next();
    }
    
    
    if ( _rowSelected == dataRow )
    {		
	// wcolor_set(stdscr, COL_CONTENT, NULL);
	wattroff(stdscr, A_REVERSE); 		
    }
}

void Pager::getAttrSpec(int colPos, const Chain& s, Chain& attr, int& len)
{    
    Tokenizer t1(s, ":");
    t1.nextToken(attr);
    Chain attrLen;
    t1.nextToken(attrLen);
    if ( attrLen == Chain("auto") )
	len = _outLen[colPos];
    else
	len = attrLen.asInteger();
}

void Pager::formatData(const ListT< ListT<Chain> > & outData)
{
    // init outLen list
    Chain *pAttr = _schema.First();
    while ( pAttr )
    {
	Tokenizer t1(*pAttr, ":");
	Chain a;
	t1.nextToken(a);
	_outLen.Insert(a.length() + 1);
	pAttr = _schema.Next();
    }

    ListT<Chain> *pRow = outData.First();
    
    int rowNo=0;
    while ( pRow )
    {
	int lineNo=0;
	
	bool moreLines=true;
	
	while ( moreLines )
	{
	    ListT<Chain> outRow;
	    
	    moreLines=false;

	    int colNum=0;
	    Chain *pCol = pRow->First();
	    while ( pCol )
	    {
		Tokenizer tok(*pCol, "\n");

		int i=0;
		Chain s;
		bool hasToken=false;
		while (  i <= lineNo )
		{
		    hasToken=tok.nextToken(s);		    		    
		    i++;
		}
		if ( hasToken )
		{
		    // cout << "Got token " << s << " for lineno = " << lineNo << endl;
		    outRow.Insert(s);

		    if ( _outLen[colNum] < s.length() + 1 )
			_outLen[colNum] = s.length() + 1;
		    
		    moreLines=true;
		}
		else
		{
		    // cout << "Got empty for lineno = " << lineNo << endl;
		    Chain empty;
		    outRow.Insert(empty);
		}
		colNum++;
		pCol = pRow->Next();
	    }
	    
	    if ( moreLines )
	    {
		_outData.Insert(outRow);
		lineNo++;
		rowNo++;
	    }
	}

	// cout << "Insert line height" << lineNo << endl;
	_rowHeight.Insert(lineNo);
	
	pRow = outData.Next();
    }
}
