///////////////////////////////////////////////////////////////////////////////
//                                                         
// File.cc
// -------
// File class implementation
//                                               
// Design and Implementation by Bjoern Lemke               
//
// (C)opyright 2000-2016 Bjoern Lemke
//
// Design and Implementation by Bjoern Lemke               
//                                                         
// IMPLEMENTATION MODULE
//
// Class: File
//
// Description: Utility class for operations on files 
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _REENTRANT
#define _REENTRANT    /* basic 3-lines for threads */
#endif

// SYSTEM INCLUDES
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#ifdef HAVE_DARWIN
#include <fcntl.h>
#endif

#include <poll.h>
#include <sys/ioctl.h>

// BASE INCLUDES
#include "Exception.h"
#include "Chain.h"
#include "File.h"
#include "Tokenizer.h"

// DEFINES

File::File()
{
    _fd = 0;
    _mode = CLOSED;
    _lineBuf = 0;
    _lineBufLen = 0;
    _readPos=0;
    _readBufLen=0;    
}

File::File(StdStream s)
{
    if ( s == IN )
    {
	_fd = STDIN_FILENO;
	_filename = Chain("stdin");
	_mode = READ;
    }
    else
    {
	_fd = STDOUT_FILENO;
	_filename = Chain("stdout");
	_mode = WRITE;
    }

    _lineBuf = 0;
    _lineBufLen = 0;
    _readPos=0;
    _readBufLen=0;    
}

File::File(char *s)
{
    _fd = 0;
    _filename = s;
    _mode = CLOSED;
    _lineBuf = 0;
    _lineBufLen = 0;
    _readPos=0;
    _readBufLen=0;    
}

File::File(const Chain&  str)
{
    _fd = 0;
    _filename = str;
    _mode = CLOSED;
    _lineBuf = 0;
    _lineBufLen = 0;
    _readPos=0;
    _readBufLen=0;
}

File::~File()
{
    if ( _lineBuf )
	free(_lineBuf);	
}

bool File::exists(const Chain& fileName)
{
    struct stat sb;
    int ret = ::stat(fileName, &sb);
    
    if ( ret == -1 ) 
    {
	if ( errno == ENOENT )
	    return false;
	else
	{
	    Chain msg = Chain("Cannot stat file ") +  fileName + Chain(" : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);	    
	}
    }
    return true;    
}

bool File::exists()
{
    struct stat sb;
    int ret = ::stat(_filename, &sb);
    
    if ( ret == -1 ) 
    {
	if ( errno == ENOENT )
	    return false;
	else
	{
	    Chain msg = Chain("Cannot stat file ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);
	    
	}
    }
    return true;
}	  

void File::open(Mode m)
{
     int oflag;
     
     switch (m)
     {
     case READ:
	 oflag = O_RDONLY;
	 break;
     case WRITE:
	 oflag = O_WRONLY | O_CREAT | O_TRUNC;
	 break;
     case READWRITE:
	 oflag = O_RDWR | O_CREAT;
	 break;
     case APPEND:
	 oflag = O_APPEND | O_WRONLY | O_CREAT;
	 break;
     case CLOSED:
	 throw Exception(EXLOC, Chain("Invalid mode for open"));
     }
     
     if (( _fd = ::open(_filename, oflag, 00644)) == -1)
     {
	 Chain msg = "Cannot open file " +  _filename;
	 throw Exception(EXLOC, msg);
     }	  

     _mode = m;
     
     return;
}

bool File::hasData(int timeout) const
{    
    struct pollfd pl[1];
    pl[0].fd = _fd;
    pl[0].events = POLLIN;

    int numfd = poll(pl, 1, timeout);    
    if ( numfd == -1 )
    {
	Chain msg = Chain("Cannot poll file ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);	
    }
	
    if ( numfd > 0 && pl[0].revents & POLLIN )
    {
	return true;
    }
    return false;
}

void File::flush()
{
     if (_mode != CLOSED)
     {

#ifdef HAVE_DARWIN
	 if ( ::fcntl(_fd, F_FULLFSYNC ) == -1 )
	 {
	     Chain msg = Chain("Cannot flush file ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	     throw Exception(EXLOC, msg);
	 }	 
#else	 
	 if ( ::fsync(_fd) == -1 )
	 {
	     Chain msg = Chain("Cannot flush file ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	     throw Exception(EXLOC, msg);
	 }	 
#endif
     }
}

void File::close()
{
     if (_mode != CLOSED)
     {
	  ::close (_fd);
	  _mode = CLOSED;
	  _fd = 0;
     }
}

void File::writeByte( char *buf, unsigned long bufsize )
{
     if ( _mode != WRITE && _mode != READWRITE && _mode != APPEND )
     {
	 Chain msg = Chain("File ") + _filename + Chain(" not open for write");
	 throw Exception(EXLOC, msg);
     }

     unsigned long numbyte = 0;

     while ( numbyte < bufsize )
     {
	 unsigned long n;
	 if ( ( n = ::write(_fd, (char*)((unsigned long long)buf + numbyte), bufsize - numbyte)) == -1 )
	 {
	     Chain msg = Chain("Write system error for ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	     throw Exception(EXLOC, msg);
	 }
	 numbyte += n;
     }
}

unsigned long File::readByte( char *buf, unsigned long bufsize )
{
     if ( _mode != READ && _mode != READWRITE )
     {
	 Chain msg = Chain("File ") + _filename + Chain(" not open for read");
	 throw Exception(EXLOC, msg );
     }

     unsigned long numbyte = 0;

     if ( (numbyte = ::read(_fd, buf, bufsize)) == -1 )
     {
	 Chain msg = Chain("Read system error for ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	 throw Exception(EXLOC, msg);
     }

     return numbyte;
}

void File::writeChain(const Chain& str)
{
     if ( _mode != WRITE && _mode != READWRITE && _mode != APPEND )
     {
	 Chain msg = Chain("File ") + _filename + Chain(" not open for write");
	 throw Exception(EXLOC, msg);
     }
     if (str.length() > 0 )
     {
	 if ( ::write(_fd, (char*)str, str.length()-1) == -1 )
	 {
	     Chain msg = Chain("Write system error for ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	     throw Exception(EXLOC, msg);
	 }
     }
     return;
}

int File::readLine(Chain& line, unsigned long long maxLen)
{
    if (_mode != READ && _mode != READWRITE )
    {
	Chain msg = Chain("File ") + _filename + Chain(" not open for read");
	throw Exception(EXLOC, msg);
    }
    
    if ( _lineBufLen < maxLen+1 )
    {
	if ( _lineBuf )
	    free(_lineBuf);	
	_lineBuf = new char[maxLen+1];
	_lineBufLen = maxLen+1;
    }

    bool eol = false;

    int i = 0;

    while ( ! eol && i < maxLen )
    {
	// if ( ::read(_fd, &_lineBuf[i], 1) == 1)
	if ( ( _lineBuf[i] = nextChar() ) != 0 )
	{
	    if ( _lineBuf[i] == '\n') 
		eol = true;
	    else if ( _lineBuf[i] == '\r')
	    {
		// windows line feed with cr and nl 
		char nl;

		nl = nextChar();
		
		// ssize_t r;
		// r = ::read(_fd, &nl, 1);

		eol = true;
	    }
	    else
		i++;
	} 
	else 
	{
	    if ( i  == 0 ) 
		return 0;
	    else
		eol = true;	       
	}
    }
    
    if ( i == maxLen )
    {
	throw Exception(EXLOC, "Line length exceeded");
    }

    _lineBuf[i] = '\0';

    line = Chain(_lineBuf);

    return i+1;
}

char File::nextChar()
{
    
    _readPos++;
    
    if ( _readPos >= _readBufLen )
    {
	_readBufLen = readByte( _readBuf, READBUFSIZE );
	if ( _readBufLen == 0 )
	    return 0;
	
	_readPos = 0;
    }

    // cout << "Char at pos " << _readPos << " is " << (int)_readBuf[_readPos] << endl;
    return _readBuf[_readPos];

}

void File::seek(unsigned long pos)
{
    if ( ::lseek(_fd, pos, SEEK_SET) == -1 )
    {
	Chain msg = Chain("Seek system error for ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    return;
}

void File::trunc(unsigned long size)
{
    if ( ::ftruncate(_fd, size) == -1 )
    {
	Chain msg = Chain("ftruncate system error for ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    return;
}

void File::rename(const Chain& fileName)
{
    if ( _mode != CLOSED )
    {
	throw Exception(EXLOC, "Closed instance is required for rename"); 
    }

    if ( ::rename((char*)_filename, (char*)fileName) == -1 )
    {
	Chain msg = Chain("Rename system error for ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    return;
}

void File::remove()
{

    if ( _mode != CLOSED )
    {
	throw Exception(EXLOC, "Closed instance is required for remove"); 
    }
    
    if ( ::unlink((char*)_filename) == -1 )
    {
	Chain msg = Chain("Unlink system error for ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    return;
}

int File::Size() const
{
    struct stat sb;

    if ( ::fstat(_fd, &sb) == -1 )
    {
	Chain msg = Chain("Fstat system error for ") +  _filename + Chain(" : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    
    return sb.st_size;   
}

const Chain& File::getFileName() const
{
    return _filename;
}

Chain File::getShortName() const
{    
    Tokenizer tok(_filename, Chain("/"));
    Chain part;
    while (tok.nextToken(part));
    return part;
}

File& File::operator = ( File& f)
{
    if ( _mode != CLOSED || f._mode != CLOSED )
    {
	throw Exception(EXLOC, "Closed instances are required for copy"); 
    }
    
    open(WRITE);
    f.open(READ);
        
    unsigned long numByte;
    char buffer[READBUFSIZE];
    	
    while ((numByte = f.readByte(buffer, READBUFSIZE)) > 0)
    {
	writeByte(buffer, numByte); 
    }
    close();
    
    return (*this);
}

File& File::operator += ( File& f)
{
    if ( _mode != CLOSED || f._mode != CLOSED )
    {
	throw Exception(EXLOC, "Closed instances are required for append"); 
    }
    
    open(APPEND);
    f.open(READ);
        
    unsigned long numByte;
    char buffer[READBUFSIZE];
    
    while ((numByte = f.readByte(buffer, READBUFSIZE)) > 0)
    {
	writeByte(buffer, numByte); 
    }
    close();

    return (*this);
}

File& File::operator << ( const Chain& str)
{
    writeChain(str);
    return (*this);
}

File& File::operator << ( unsigned long l)
{
    Chain s(l);
    writeChain(s);
    return (*this);
}

File& File::operator << ( int i)
{
    Chain s(i);
    writeChain(s);
    return (*this);
}

File& File::operator << ( char c)
{
    char buf[2];
    buf[0] = c;
    buf[1] = 0;

    writeChain(buf);
    return (*this);
}
