///////////////////////////////////////////////////////////////////////////////
//                                                         
// BigInteger.cc
// -------------
// BigInteger implementation
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//                                                         
// IMPLEMENTATION MODULE
//
// Class: BigInteger
// 
// Description: BigInteger implementation
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// INCLUDES
#include "Exception.h"
#include "BigInteger.h"

BigInteger::BigInteger()
{
    _val = Chain(0); 
    _isPositive = true;
}

BigInteger::BigInteger(const BigInteger& bi)
{
    _val = bi._val;
    _isPositive = bi._isPositive;
}

BigInteger::BigInteger(const Chain& val)
{
    if ( val.subChain(1,1) == Chain("+"))
    {
	_isPositive=true;
	_val = val.subChain(2, val.length());
    }
    else if ( val.subChain(1,1) == Chain("-"))
    {
	_isPositive=false;
	_val = val.subChain(2, val.length());
    }
    else
    {
	_isPositive=true;
	_val = val.truncLeft(Chain("0"));
	if ( _val.length() <= 1 )
	    _val = Chain(0);
    }

    if ( ! _val.isNum() )
    {
	Chain msg = Chain("Invalid number format for <") + val + Chain(">");
	throw Exception(EXLOC, msg);
    }
}

BigInteger::~BigInteger()
{
}

BigInteger BigInteger::abs() const
{
    return BigInteger(_val);
}

bool BigInteger::isPositive() const
{
    return _isPositive;
}

void BigInteger::negate()
{
    _isPositive = false;
}

BigInteger BigInteger::add(const BigInteger& d) const
{
    if ( ( isPositive() == true && d.isPositive() == true )
	 || ( isPositive() == false  && d.isPositive() == false) )
    {
	int l = length() < d.length() ? d.length() : length();
	
	int carry = 0;
	int rest = 0;
	
	Chain r;
	for ( int i = 1 ; i <= l ; i++ )
	{
	    int j = getDigit(length() - i) + d.getDigit(d.length() - i) + carry;
	    
	    rest = j % 10;
	    carry = j / 10;
	    
	    r = Chain(rest) + r;
	}
	
	BigInteger res;
	res = BigInteger(r.truncLeft(Chain(0)));
	
	if ( _isPositive == false)
	    res.negate();
	return res;
    }
    else if ( isPositive() == true )
    {
	return sub( BigInteger(d._val) );
    }
    else
    {
	return d.sub( BigInteger(_val) );
    }
}

BigInteger BigInteger::sub(const BigInteger& d) const
{
    if ( isPositive() == true && d.isPositive() == true)
    {
	if ( *this >= d )
	{
	    int l = length() < d.length() ? d.length() : length();
	    
	    int carry = 0;
	    int rest = 0;
	    
	    Chain r;
	    for ( int i = 1 ; i <= l ; i++ )
	    {
		
		int j = getDigit(length() - i) - d.getDigit(d.length() - i) - carry;
		if ( j < 0 )
		{
		    rest = 10 + j;
		    carry = 1 - ( (j+1) / 10 );
		}
		else
		{
		    rest = j;
		    carry = 0;
		}
		
		r = Chain(rest) + r;
	    }
	   	    
	    BigInteger res(r.truncLeft(Chain(0)));
	    return BigInteger(res);
	}
	else
	{
	    BigInteger s = d.sub(*this);
	    s.negate();
	    return s;
	}
    }
    else if ( isPositive() == true && d.isPositive() == false)
    {
	return add(d);
    }
    else if ( isPositive() == false && d.isPositive() == true)
    {
	BigInteger s(_val);
        s = s.add(d);
	s.negate();
	return s;
    }
    else // if ( isPositive() == false && d.isPositive() == false)
    {
	BigInteger s(d._val);	
	BigInteger t(_val);
        s = s.sub(t);
	return s;
    }
}

BigInteger BigInteger::mul(const BigInteger& d) const
{
 
    BigInteger res;
    for ( int i = 1 ; i < d.length() ; i ++ )
    {
	BigInteger s = mulDigit(d.getDigit(d.length() - i));
	if ( i == 1 )
	{
	    res = s;
	}
	else
	{
	    int j;
	    Chain n = s.toChain();
	    for ( j=1; j<i ; j++)
		n = n + Chain(0);
	    BigInteger a(n);
	    res = res.add(a);   
	}
    }

    if ( ( isPositive() == true && d.isPositive() == false )
	 || ( isPositive() == false && d.isPositive() == true ))
	res.negate();
    return res;
}

BigInteger BigInteger::mulDigit(int f) const
{     
    Chain d;
    int carry = 0;
    int rest = 0;
    for ( int i=1 ; i <= length() ; i++ )
    {
	int m = getDigit(length() - i) * f + carry;
	rest = m % 10;
	carry = m / 10;
	d = Chain(rest) + d;
    }

    BigInteger res(d.truncLeft(Chain(0)));
    return res;
}

BigInteger BigInteger::div(const BigInteger& d) const
{
    if ( *this == d )
	return BigInteger(1);

    if ( *this < d )
	return BigInteger(0);

    BigInteger divPart;
    int j = 1;
    Chain res;
    while ( j < (int)_val.length() )
    {
	
	divPart = BigInteger( divPart.toChain() + _val.subChain(j,j) );
	// cout << "DivPart=" << divPart << endl;

	j++;
	while ( divPart < d && j <= (int)_val.length() ) 
	{

	    divPart = BigInteger( divPart.toChain() + _val.subChain(j,j) );
	    // cout << "divPart ->" << divPart << endl;
	    j++;
	    res = res + Chain(0);
	    // cout << "Res=" << res << endl;
	}

	if ( j <= (int)_val.length() )
	{
	    BigInteger s;
	    int mod=0;
	    while ( s < divPart ) 
	    {
		mod++;
		s = s.add(d);
	    }
	    if ( s > divPart )
		mod--;
	    
	    while ( divPart >= d )
	    {
		divPart = divPart.sub(d);
	    }
	    
	    // cout << "dp=" << divPart<< endl;
	    // cout << "mod=" << mod << endl;
	    res = res + Chain(mod);
	    // cout << "Res=" << res << endl;
	}
    }

    BigInteger dres(res.truncLeft(Chain(0)));
    
    if ( ( isPositive() == true && d.isPositive() == false )
	 || ( isPositive() == false && d.isPositive() == true ) )
	dres.negate();
    
    
    return dres; 
}

int BigInteger::length() const
{
    return (int)_val.length();
}

int BigInteger::getDigit(int i) const
{
    if ( i < length() && i > 0)
	return _val.subChain(i, i).asInteger();
    else
	return 0;
}

Chain BigInteger::toChain() const
{
    Chain s;
    if ( _isPositive == false )
	s = Chain("-");
    return s + _val;
}

BigInteger& BigInteger::operator = ( const BigInteger& d)
{
    _val = d._val;
    _isPositive = d._isPositive;
    return (*this);
}

BigInteger& BigInteger::operator += ( const BigInteger& d)
{
    this->add(d);
    return (*this);
}

BigInteger& BigInteger::operator -= ( const BigInteger& d)
{
    this->add(d);
    return (*this);
}

bool BigInteger::operator == ( const BigInteger& d) const
{
    return ( _val == d._val && _isPositive == d._isPositive );
}

bool BigInteger::operator != ( const BigInteger& d) const
{
    return ( _val != d._val || _isPositive != d._isPositive );
}

bool BigInteger::operator < ( const BigInteger& d) const
{
    if ( _isPositive == false && d._isPositive == true )
	return true;
    else if ( _isPositive == true && d._isPositive == false )
	return false;
    else if ( _isPositive == true && d._isPositive == true )
    {
	if ( length() < d.length() )
	{
	    return true;
	}
	else if ( length() > d.length() )
	{
	    return false;
	}
	else
	{
	    for ( int i=1 ; i<length() ; i++ )
	    {
		if ( getDigit(i) < d.getDigit(i) )
		    return true;
		else if ( getDigit(i) > d.getDigit(i) )
		    return false;
	    }
	    return false;
	}
    }
    else
    {
	return BigInteger(d._val) < BigInteger(_val);
    }
}

bool BigInteger::operator > ( const BigInteger& d) const
{
    if ( _isPositive == false && d._isPositive == true )
	return false;
    else if ( _isPositive == true && d._isPositive == false )
	return true;
    else if ( _isPositive == true && d._isPositive == true )
    {

	if ( length() > d.length() )
	{
	    return true;
	}
	else if ( length() < d.length() )
	{
	    return false;
	}
	else
	{
	    for ( int i=1 ; i<length() ; i++ )
	    {
		if ( getDigit(i) > d.getDigit(i) )
		    return true;
		else if ( getDigit(i) < d.getDigit(i) )
		    return false;
	    }
	    return false;
	}
    }
    else
    {
	return BigInteger(d._val) > BigInteger(_val);
    }
}

bool BigInteger::operator <= ( const BigInteger& d) const
{
    if ( _val == d._val && _isPositive == d._isPositive)
	return true;
    else
	return *this < d;	
}

bool BigInteger::operator >= ( const BigInteger& d) const
{
    if ( _val == d._val && _isPositive == d._isPositive)
	return true;
    else
	return *this > d;
}

BigInteger operator + ( const BigInteger& d1, const BigInteger& d2)
{
    return d1.add(d2);
}

BigInteger operator - ( const BigInteger& d1, const BigInteger& d2)
{
    return d1.sub(d2);
}

BigInteger operator * ( const BigInteger& d1, const BigInteger& d2)
{
    return d1.mul(d2);
}

BigInteger operator / ( const BigInteger& d1, const BigInteger& d2)
{
    return d1.div(d2);
}

ostream& operator << (ostream& s, const  BigInteger& d)
{
    if ( d._isPositive )
	cout << d._val;
    else
	cout << "-" << d._val;
    return s;
}
