/////////////////////////////////////////////////////////////////////////////// 
// 
// Screen.cc 
// ---------- 
// Curses based screen implementation 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2017-2022 Bjoern Lemke 
// 
// IMPLEMENTATION MODULE 
// 
// Class: Screen 
// 
// Description: Cursor based curses screen implementation 
// 
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 


#include <stdio.h> 
#include <stdlib.h> 
#include <unistd.h> 

#include "Screen.h"
#include "Tokenizer.h"
#include "Sleeper.h" 

#define COL_HEADER_PASSIVE 1 
#define COL_HEADER_ACTIVE 2 
#define COL_TITLE 3 
#define COL_CONTENT 4
#define COL_SELECTED 5

// advanced color map
#define COL_BLUE 6
#define COL_GREEN 7
#define COL_YELLOW 8
#define COL_RED 9
#define COL_INPUT 10

#define LEFTOFFSET 1 
#define TOPOFFSET 1 

#define KEY_TAB 9 
#define KEY_RETURN 10 
#define KEY_ESC 27 
#define KEY_DEL 127 

#define INPUT_TYPE_STRING "S" 
#define INPUT_TYPE_ID "I" 
#define INPUT_TYPE_NUMBER "N" 
#define INPUT_TYPE_PWD "P"
#define INPUT_TYPE_MENU "M" 

#define CUR_ROW_OFFSET 3 
#define CUR_COL_OFFSET 20 
#define VAL_MAX_LEN 100 
#define VAL_MAX_NUM 20 
#define OK_COL_POS 2
#define ABORT_COL_POS 9

#define MSEC_SLEEPDELAY 100

Screen::Screen(const Chain& attrSepToken, const Chain& listSepToken, const Chain& valSepToken) : SigHandler() 
{ 
    _attrSepToken = attrSepToken;
    _listSepToken = listSepToken;
    _valSepToken = valSepToken;
    
    _menuSelected = 0;
    
#ifndef LFCNOCURSES

    /*  Initialize ncurses  */ 
    initscr();
    curs_set(0); 
    start_color(); 

    init_pair(COL_HEADER_PASSIVE,  COLOR_WHITE,     COLOR_BLACK); 
    init_pair(COL_HEADER_ACTIVE,  COLOR_YELLOW,     COLOR_BLACK); 
    init_pair(COL_TITLE,  COLOR_YELLOW,     COLOR_BLACK); 
    init_pair(COL_CONTENT,  COLOR_YELLOW,     COLOR_BLACK);
    init_pair(COL_SELECTED,  COLOR_GREEN,     COLOR_BLACK);

    init_pair(COL_INPUT,  COLOR_GREEN,  COLOR_BLACK);

    init_pair(COL_BLUE,  COLOR_BLUE,     COLOR_BLACK);
    init_pair(COL_RED,  COLOR_RED,     COLOR_BLACK);
    init_pair(COL_YELLOW,  COLOR_YELLOW,     COLOR_BLACK);
    init_pair(COL_GREEN,  COLOR_GREEN,     COLOR_BLACK); 
    
    noecho();                  /*  Turn off key echoing                 */ 
    keypad(stdscr, TRUE);
    // keypad(_mainwin, TRUE);     /*  Enable the keypad for non-char keys  */ 
    
    timeout(0); 

    init();

    _statusLine=Chain();
    _statusOffset=0;
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

#ifndef HAVE_MINGW 
    install(SIGWINCH); 
#endif
    
} 

Screen::~Screen() 
{
#ifndef LFCNOCURSES
    delwin(stdscr); 
    endwin(); 
    refresh();
#endif
} 


void Screen::sigCatch(int sig) 
{ 

    try 
    {	
#ifndef LFCNOCURSES
	
        clear();
	
	endwin();	
	initscr();

	// Sleeper::milliSleep(MSEC_SLEEPDELAY);
	
	// Needs to be called after an endwin() so ncurses will initialize 
	// itself with the new terminal dimensions. 
	// refresh(); 
	clear();        		
	refresh();

#endif
	// we nee to add a delay, otherwise signal handling can lead to core dumps
	

#ifndef HAVE_MINGW 
	install(SIGWINCH);
#endif
	
    } 
    catch ( Exception e) 
    { 
        Chain msg; 
        e.pop(msg); 
        cout << msg << endl; 
    }

    
}; 


void Screen::regMenu(const Chain& name, int id) 
{ 
    _menu.Insert(MenuItem(name, id)); 
} 

void Screen::regShortCut(char c, int code) 
{ 
    _scList.Insert(ShortCut(c, code)); 
}

int Screen::showHeader() 
{
#ifndef LFCNOCURSES
    // clear(); 
    
    bool isSelected=false; 

    while( isSelected == false ) 
    { 
	bool doRefresh = true;

	wtimeout(stdscr, MSEC_SLEEPDELAY);
        int c = wgetch(stdscr); 

        switch(c) 
        { 
        case KEY_RIGHT: 
            if ( _menuSelected < _menu.Size()-1 ) 
                _menuSelected++; 
            else 
                _menuSelected = 0;
            break; 
        case KEY_LEFT: 
            if ( _menuSelected > 0  ) 
                _menuSelected--; 
            else 
                _menuSelected = _menu.Size()-1; 
            break; 
        case KEY_RETURN: 
        { 
            isSelected = true; 
            break; 
        }
	case ERR:
	{
	    // mvwprintw(stdscr, 1, 1, "Timeout %d", c); 
	    // timeout
	    break;
	}
	default:
	{
	    doRefresh=false;
	    break;
        }
        } 

	if ( doRefresh )
	    refreshFrame(); 

	// refreshFooter("Main menu");
    }
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return _menu[_menuSelected].getId(); 
} 

void Screen::refreshFrame() 
{
    
#ifndef LFCNOCURSES

    clear();
   
    int colno=LEFTOFFSET; 
    attroff(A_REVERSE); 

    MenuItem *pMI = _menu.First(); 
    int idx=0; 
    while ( pMI ) 
    {
	
        if ( _menuSelected == idx ) 
        { 
            attron(A_REVERSE); 
            color_set(COL_HEADER_ACTIVE, NULL); 
        } 
        else 
        { 
            color_set(COL_HEADER_PASSIVE, NULL); 
        }
	// mvwprintw(stdscr, 1, 1, (char*)Chain(COLS)) ; 
        mvprintw(0, colno, "%s", (char*)pMI->getName()); 
        colno += pMI->getName().length() + 1; 
        attroff(A_REVERSE); 
	
        idx++; 
        pMI=_menu.Next(); 
    }

    char *status = (char*)_statusLine;
    if ( status != 0 )
    {
	int mainy, mainx;   
	getmaxyx(stdscr, mainy, mainx);
	mvwprintw(stdscr, mainy-1, _statusOffset, "%s", status);
	wmove(stdscr, mainy-1, 0); 
    }

    
    refresh();
#endif
}


void Screen::showInfoBox(const Chain& title, const Chain& msg, int width) 
{ 
#ifndef LFCNOCURSES

    int height=0; 

    Tokenizer t(msg, Chain("\n"));
    Chain l;
    while ( t.nextToken(l))
    {
	height++;
    }

    height+=4;

    int c;

    int startx = (COLS - width) / 2;
    int starty = (LINES - height) / 2;

    WINDOW *infowin = newwin(height, width, starty, startx);

    noecho();

    keypad(infowin, TRUE);

    wattron(infowin, A_BOLD);
    mvwprintw(infowin, 1, 2, "%s", (char*)title);
    wattroff(infowin, A_BOLD);
    box(infowin, 0, 0);

    t.reset();
    int lno=3;
    while ( t.nextToken(l))
    {
	if ( (int)l.length() + 4 > width )
	    l = l.subChain(1, width - 4);
	mvwprintw(infowin, lno, 2, "%s", (char*)l);
	lno++;
    }
    
    // any input to continue 

    c = wgetch(infowin); 
    switch(c) 
    { 
    case 'Y': 
        break; 
    case 'N': 
        break; 
    } 

    delwin(infowin);
    
    refreshFrame();
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

}

int Screen::showConfirmBox(const Chain& title, const Chain& details, int posx, int posy)
{
#ifndef LFCNOCURSES
    
    int width = max(title.length(), details.length()) + 5; 
    int height = 7;
        
    bool showIt=true;

    WINDOW* confwin = 0;

    int curRow = 5; 
    int curCol = OK_COL_POS; 
    
    while( showIt )
    {

	int startx = posx; 
	int starty = posy; 
	if ( startx == 0 && starty == 0 ) 
	{
	    if ( COLS >= width ) 
		startx = (COLS - width) / 2;
	    if ( LINES >= height ) 
		starty = (LINES - height) / 2; 
	}
	
	if ( confwin == 0 )
	    confwin = newwin(height, width, starty, startx);

	keypad(confwin, TRUE);

	wattron(confwin, A_BOLD);
	mvwprintw(confwin, 1, 2, "%s", (char*)title);
	wattroff(confwin, A_BOLD);

	if ( details != Chain() )
	{
	    mvwprintw(confwin, 3, 2, "%s", (char*)details);
	}

	box(confwin, 0, 0); 

	curs_set(0);
	if ( curCol == OK_COL_POS )
	{
	    wattron(confwin, A_REVERSE);
	    mvwprintw(confwin, curRow, curCol, "OK");
	    wattroff(confwin, A_REVERSE);
	    mvwprintw(confwin, curRow, curCol+7, "Abort");
	}
	else if ( curCol == ABORT_COL_POS )
	{
	    mvwprintw(confwin, curRow, curCol-7, "OK");
	    wattron(confwin, A_REVERSE);
	    mvwprintw(confwin, curRow, curCol, "Abort");
	    wattroff(confwin, A_REVERSE);
	}

	wmove(confwin, curRow, curCol); 

	wrefresh(confwin);
	noecho(); 

	wtimeout(confwin, MSEC_SLEEPDELAY);
	int c = wgetch(confwin);
	switch(c)
	{	
	case KEY_RESIZE:
	{
	    delwin(confwin);
	    refreshFrame();
	    confwin=0;
	    break;  
	}
	case KEY_TAB:
	case KEY_DOWN:
	case KEY_UP:
	case KEY_LEFT:
	case KEY_RIGHT:
	    if ( curCol == OK_COL_POS )
		curCol = ABORT_COL_POS;
	    else if ( curCol == ABORT_COL_POS )
		curCol = OK_COL_POS;
            wmove(confwin, curRow, curCol); 
            break; 
	    
	    break;
	case KEY_RETURN:
	{	    
	    showIt = false;
	    break;
	}
	case KEY_ESC:
	{
	    showIt = false;
	    break;
	}
	}
    }

    if ( confwin )
	delwin(confwin);

    refreshFrame();
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    if ( curCol == OK_COL_POS )
	return 1;
    return 0;

}

int Screen::showSelectBox(const Chain& title, const ListT<Chain>& selectList,
			   Chain& selectKey,
			   Chain& selectValue,
			   int posx, int posy)
{
#ifndef LFCNOCURSES

    int width = title.length() + 5;
    Chain *pS = selectList.First();
    while ( pS )
    {
	width = (int)pS->length() + 1 > width ? (int)pS->length() + 1 : width;   
	pS = selectList.Next();
    }
    int height = selectList.Size() + 4;
 
    int highlight=1;
    int choice = 0;
    int c;
    
	
    bool showIt=true;

    WINDOW* menuwin = 0;
    
    while( showIt )
    {

	int startx = posx; 
	int starty = posy; 
	if ( startx == 0 && starty == 0 ) 
	{
	    if ( COLS >= width ) 
		startx = (COLS - width) / 2;
	    if ( LINES >= height ) 
		starty = (LINES - height) / 2; 
	}
	
	if ( menuwin == 0 )
	    menuwin = newwin(height, width, starty, startx);

	keypad(menuwin, TRUE);
	printMenu(menuwin, title, selectList, highlight);
    
	noecho(); 

	wtimeout(menuwin, MSEC_SLEEPDELAY); 	
	c = wgetch(menuwin);
	switch(c)
	{	
	case KEY_UP:
	{
	    if (highlight == 1)
		highlight = selectList.Size();
	    else
		--highlight;
	    break;
	}
	case KEY_DOWN:
	{
	    if(highlight == selectList.Size() )
		highlight = 1;
	    else 
		++highlight;
	    break;
	}
	case KEY_RESIZE:
	{
	    delwin(menuwin);
	    refreshFrame();
	    menuwin=0;
	    break;  
	}

	case KEY_RETURN:
	{
	    showIt = false;
	    choice = highlight;
	    break;
	}
	case KEY_ESC:
	{
	    showIt = false;
	    break;
	}

	}
    }

    if ( menuwin )
	delwin(menuwin);

    refreshFrame();
    
    if ( choice > 0 )
    {
	getKeyValue(selectList[highlight-1], selectKey, selectValue);
	return choice;
    }

#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return 0;	
}

#ifndef LFCNOCURSES

void Screen::printMenu(WINDOW* win, const Chain& title, const ListT<Chain>& menuList, int highlight)
{
    wattron(win, A_BOLD);
    mvwprintw(win, 1, 2, "%s", (char*)title);
    wattroff(win, A_BOLD);
    int x = 2;
    int y = 3;
    box(win, 0, 0);
    for(int i = 0; i < menuList.Size(); ++i)
    {	
	Chain key, value;
	getKeyValue(menuList[i], key, value);
	
	if (highlight == i + 1) /* High light the present choice */
	{	
	    wattron(win, A_REVERSE);
	    mvwprintw(win, y, x, "%s", (char*)key);
	    wattroff(win, A_REVERSE);
	}
	else
	    mvwprintw(win, y, x, "%s", (char*)key);
	++y;
    }

    

    wrefresh(win);
}
#endif


int Screen::showFormBox(const Chain& title, const ListT<Chain>& attrList, ListT<Chain>& valList, int maxVisible)
{
    int returnVal = 0;

#ifndef LFCNOCURSES

    int height = attrList.Size() + CUR_ROW_OFFSET + 3;

    int c;
    char inputArray[VAL_MAX_NUM][VAL_MAX_LEN];
    
    int maxLenArray[VAL_MAX_NUM];
    int vposArray[VAL_MAX_NUM];
    
    // init array
    for(int i = 0; i < VAL_MAX_NUM; i++)
    {
	for(int j = 0; j < VAL_MAX_LEN; j++)
	    inputArray[i][j]=0;
	vposArray[i]=0;
    }

    int colOffset = 0;
    int width = 0;
    
    for(int i = 0; i < attrList.Size(); ++i)
    {
	Chain attr, type, value;
	int maxLen;
	getAttrTypeValue(attrList[i], attr, type, maxLen, value);
	maxLenArray[i] = maxLen;
	
	if ( (char*)value )
	{	    
	    if ( type == Chain(INPUT_TYPE_MENU) )
	    {
		Tokenizer t1(value, _listSepToken);
		Chain menuItem;
		// first token is active value
		if ( t1.nextToken(menuItem))
		{
		    Tokenizer t2(menuItem, _valSepToken);
		    Chain v;
		    if ( t2.nextToken(v))
			value = v;
		}
	    }
	    
            for(int j = 0; j < (int)value.length(); j++)
                inputArray[i][j]=value[j];
            for(int j = (int)value.length(); j < VAL_MAX_LEN; j++)
                inputArray[i][j]=0;
	}
	else
	{
	    inputArray[i][0]=0;
	}

	if ( colOffset < (int)attr.length() )
	    colOffset = attr.length();
	
	if ( width < maxVisible + (int)attr.length() )
	    width = maxVisible + attr.length();
    }
    
    width+=8;
    colOffset+=4;
    
    int curRow = CUR_ROW_OFFSET;
    int curCol = colOffset;
    
    bool showMask=true;
    
    WINDOW *maskwin = 0;
    
    int isFirst=1;
    while( showMask )
    {
	
	int startx = 0, starty = 0;
	if ( COLS >= width )
	    startx = (COLS - width) / 2;
	if ( LINES >= height )
	    starty = (LINES - height) / 2;
	
	if ( maskwin == 0 )
	    maskwin = newwin(height, width, starty, startx);
	
	noecho();
	
	wcolor_set(maskwin, COL_CONTENT, NULL);
	keypad(maskwin, TRUE);
	
	wattron(maskwin, A_BOLD);
	mvwprintw(maskwin, 1, 2, "%s", (char*)title);
	wattroff(maskwin, A_BOLD);
	int x = 2;
	int y = CUR_ROW_OFFSET;
	box(maskwin, 0, 0);
	for(int i = 0; i < attrList.Size(); ++i)
	{
	    
	    Chain attr, type, value;
	    int maxLen;
	    getAttrTypeValue(attrList[i], attr, type, maxLen, value);
	    
	    mvwprintw(maskwin, y, x, "%s", (char*)attr);
	    mvwprintw(maskwin, y, colOffset - 2, ":");
	    
	    if ( type == Chain(INPUT_TYPE_MENU) )
	    {
		wcolor_set(maskwin, COL_TITLE, NULL);
		
		wattron(maskwin, A_BOLD);
		
		// cleanup first
		int j = 0;
		while ( j < maxLen )
		{
		    mvwprintw(maskwin, y, colOffset + j, "%c", ' ');
		    j++;
		}
		
		mvwprintw(maskwin, y, colOffset, "%s", (char*)inputArray[i]);
		
		wattroff(maskwin, A_BOLD);
		wcolor_set(maskwin, COL_CONTENT, NULL);
	    }
	    else
	    {
		
		wattron(maskwin, A_UNDERLINE);
		wcolor_set(maskwin, COL_INPUT, NULL);
		for(int j = 0; j < min(maxLen,maxVisible); ++j)
		    mvwprintw(maskwin, y, colOffset + j, " ");
		if ( inputArray[i][0] )
		{
		    if ( type == Chain(INPUT_TYPE_PWD) )
		    {
			int j=0;
			while ( inputArray[i][j] != 0 )
			{
			    mvwprintw(maskwin, y, colOffset + j, "%c", '*');
			    j++;
			}
		    }
		    else
		    {
			int j=vposArray[y - CUR_ROW_OFFSET];			
			int pos=0;
			while ( inputArray[i][j] != 0 && j < vposArray[y - CUR_ROW_OFFSET] + maxVisible )
			{
			    mvwprintw(maskwin, y, colOffset+pos, "%c", inputArray[i][j]);
			    j++;
			    pos++;
			}			
		    }
		}
		
		wattroff(maskwin, A_UNDERLINE);
		wcolor_set(maskwin, COL_CONTENT, NULL);
	    }
	    
	    ++y;
	}
	y++;
	
	if ( isFirst )
	{
	    wattron(maskwin, A_BOLD);
	    mvwprintw(maskwin, y, x, "Ok     Abort");
	    wattroff(maskwin, A_BOLD);
	    isFirst=0;
	}
	wmove(maskwin, curRow, curCol);
	
	wtimeout(maskwin, MSEC_SLEEPDELAY);
        c = wgetch(maskwin);
	
	switch(c)
        {
        case KEY_UP:
            int prevCurRow, prevCurCol;
            prevCursorPos(attrList.Size(), colOffset, curRow, curCol, prevCurRow, prevCurCol);
            curRow = prevCurRow;
            curCol = prevCurCol;
            wmove(maskwin, curRow, curCol);
            break;
        case KEY_DOWN:
	case KEY_TAB:
            int nextCurRow, nextCurCol;
	    nextCursorPos(attrList.Size(), colOffset, curRow, curCol, nextCurRow, nextCurCol, 0);
            curRow = nextCurRow;
            curCol = nextCurCol;
            wmove(maskwin, curRow, curCol);
            break;
        case KEY_LEFT:
            if ( curRow < attrList.Size() + CUR_ROW_OFFSET
		 &&  vposArray[curRow - CUR_ROW_OFFSET] > 0 )
	    {
		vposArray[curRow - CUR_ROW_OFFSET]--;
	    }
            else if ( curRow < attrList.Size() + CUR_ROW_OFFSET
		      && curCol > colOffset )
	    {
                curCol--;
	    }
	    else if ( curRow == attrList.Size() + CUR_ROW_OFFSET + 1 )
	    {
		int prevCurRow, prevCurCol;
		prevCursorPos(attrList.Size(), colOffset, curRow, curCol, prevCurRow, prevCurCol);
		curRow = prevCurRow;
		curCol = prevCurCol;
	    }
	    wmove(maskwin, curRow, curCol);
            break;
        case KEY_RIGHT:
	{	  	    
            if ( curRow < attrList.Size() + CUR_ROW_OFFSET
		 && inputArray[curRow - CUR_ROW_OFFSET][curCol - colOffset + vposArray[curRow - CUR_ROW_OFFSET]] != 0)
	    {
		if ( curCol < colOffset + maxVisible )
		{
		    curCol++;
		}
		else if ( curRow < attrList.Size() + CUR_ROW_OFFSET
			  && curCol >= colOffset + maxVisible
			  && vposArray[curRow - CUR_ROW_OFFSET] + maxVisible < maxLenArray[curRow - CUR_ROW_OFFSET] )
		{
		    vposArray[curRow - CUR_ROW_OFFSET]++;
		    setStatusLine(Chain(vposArray[curRow - CUR_ROW_OFFSET]));
		}
	    }
	    else if ( curRow == attrList.Size() + CUR_ROW_OFFSET + 1 )
	    {
		int prevCurRow, prevCurCol;
		nextCursorPos(attrList.Size(), colOffset, curRow, curCol, prevCurRow, prevCurCol, 0);
		curRow = prevCurRow;
		curCol = prevCurCol;
	    }
	    wmove(maskwin, curRow, curCol);
            break;
	}
        case KEY_RETURN:
        {
            if ( curRow == attrList.Size() + CUR_ROW_OFFSET + 1 )
            {
                if ( curCol == OK_COL_POS )
                    returnVal = 1;
                else if ( curCol == ABORT_COL_POS )
                    returnVal = 0;
                showMask=false;
            }
	    else
            {
                Chain attr, type, value;
		int maxLen;
                getAttrTypeValue(attrList[curRow-CUR_ROW_OFFSET], attr, type, maxLen, value);
                if ( type == Chain(INPUT_TYPE_MENU))
                {		    
		    Tokenizer t(value, _listSepToken);
		    
		    ListT<Chain> menu;
		    Chain menuItem;
		    
		    // first value is just active value
		    t.nextToken(menuItem);
		    
		    while ( t.nextToken(menuItem))
		    {
			menu.Insert(menuItem);
		    }
		    
		    Chain selectKey;
		    Chain selectValue;
		    
		    int r = showSelectBox("Selection", menu,
					  selectKey,
					  selectValue,
					  startx + colOffset, starty + curRow);
		    
		    if ( r )
		    {
			if ( (char*)selectKey )
			{
			    for(int j = 0; j < (int)selectKey.length(); j++)
				inputArray[curRow-CUR_ROW_OFFSET][j]=selectKey[j];
			    for(int j = (int)selectKey.length(); j < VAL_MAX_LEN; j++)
				inputArray[curRow-CUR_ROW_OFFSET][j]=0;
			}
			else
			{
			    inputArray[curRow-CUR_ROW_OFFSET][0]=0;
			}
		    }
                }
		else
		{
		    int nextCurRow, nextCurCol;
		    nextCursorPos(attrList.Size(), colOffset, curRow, curCol, nextCurRow, nextCurCol, 0);
		    curRow = nextCurRow;
		    curCol = nextCurCol;
		    wmove(maskwin, curRow, curCol);
		}
            }
            break;
        }
        case KEY_DEL:
	case KEY_BACKSPACE:
        {
	    Chain attr, type, value;
	    int maxLen;
	    getAttrTypeValue(attrList[curRow-CUR_ROW_OFFSET], attr, type, maxLen, value);
	    if ( type != Chain(INPUT_TYPE_MENU))
	    {
		if ( curCol > colOffset && vposArray[curRow - CUR_ROW_OFFSET] == 0 )
		    curCol--;
		else if ( vposArray[curRow - CUR_ROW_OFFSET] > 0 )
		    vposArray[curRow - CUR_ROW_OFFSET]--;
		
		for ( int i=curCol-colOffset + vposArray[curRow - CUR_ROW_OFFSET];
		      i<VAL_MAX_LEN;
		      i++)
		    inputArray[curRow-CUR_ROW_OFFSET][i] = inputArray[curRow-CUR_ROW_OFFSET][i+1];
		wmove(maskwin, curRow, curCol);
	    }
	    break;
	}
        case KEY_ESC:
        {
            returnVal = 0;
            showMask=false;
            break;
        }
	case KEY_RESIZE:
	{
	    delwin(maskwin);
	    refreshFrame();
	    maskwin=0;
	    break;
	}
	case ERR:
	{
	    // timeout
	    break;
	}
        default:
        {
            if ( curRow < attrList.Size() + CUR_ROW_OFFSET )
            {		
		Chain attr, type, value;
		int maxLen;
		getAttrTypeValue(attrList[curRow-CUR_ROW_OFFSET], attr, type, maxLen, value);
		
                bool inputAccepted=false;
		
		if ( vposArray[curRow-CUR_ROW_OFFSET] + curCol - colOffset < maxLen )
		{
		    if ( type == Chain(INPUT_TYPE_STRING)
			 && ( isalnum (c)
			      || c == '_'
			      || c == '-'
			      || c == '/'
			      || c == '.'
			      || c == ','
			      || c == ';'
			      || c == '*'
			      || c == '['
			      || c == ']'
			      || c == ':'
			      || c == '@'
			      || c == '='
			      || c == '!'
			      || c == '?'
			      || c == '*'
			      || c == '+'
			      || c == '#'
			      || c == '('
			      || c == ')'
			      || c == '{'
			      || c == '}'
			      || c == '['
			      || c == ']'
			      || c == '%'
			      || c == '&'
			      || c == '$'
			      || c == '>'
			      || c == '<'
			      || c == ' ' ))
			inputAccepted=true;
		    else if ( type == Chain(INPUT_TYPE_ID)
			      && ( isalnum (c)
				   || c == '_' ))
			inputAccepted=true;
		    else if ( type == Chain(INPUT_TYPE_NUMBER) && isdigit (c) )
			inputAccepted=true;
		    else if ( type == Chain(INPUT_TYPE_PWD) )
			inputAccepted=true;
		}
		
                if ( inputAccepted )
                {		    

		    for ( int i=VAL_MAX_LEN -1;
			  i >= vposArray[curRow-CUR_ROW_OFFSET] + curCol - colOffset;
			  i--)
			inputArray[curRow-CUR_ROW_OFFSET][i] = inputArray[curRow-CUR_ROW_OFFSET][i-1];
		    
		    
		    inputArray[curRow-CUR_ROW_OFFSET][vposArray[curRow-CUR_ROW_OFFSET] + curCol - colOffset] = c;
		    
		    
		    if (  curCol >= colOffset + maxVisible ) // maxLenArray[curRow - CUR_ROW_OFFSET] )
			vposArray[curRow-CUR_ROW_OFFSET]++;
		    else
			curCol++;
		    wmove(maskwin, curRow, curCol);
                }
            }
	    break;
        }
        }
	
	if ( curRow == attrList.Size() + CUR_ROW_OFFSET + 1 )
        {
	    curs_set(0);
            if ( curCol == OK_COL_POS )
            {
                wattron(maskwin, A_REVERSE);
                mvwprintw(maskwin, curRow, curCol, "OK");
                wattroff(maskwin, A_REVERSE);
                mvwprintw(maskwin, curRow, curCol+7, "Abort");
            }
            else if ( curCol == ABORT_COL_POS )
            {
                mvwprintw(maskwin, curRow, curCol-7, "OK");
                wattron(maskwin, A_REVERSE);
                mvwprintw(maskwin, curRow, curCol, "Abort");
                wattroff(maskwin, A_REVERSE);
            }
        }
        else
        {
	    curs_set(1);
            wattron(maskwin, A_BOLD);
            mvwprintw(maskwin, y, x, "Ok     Abort");
            wattroff(maskwin, A_BOLD);	    	    
        }
	
        wrefresh(maskwin);
    }

    curs_set(0);

    if ( maskwin )
    delwin(maskwin);

    refreshFrame();

    // store values in valList
    
    valList.Empty();
    for(int i = 0; i < attrList.Size(); ++i)
    {
	Chain attr, type, value;
	int maxLen;
	getAttrTypeValue(attrList[i], attr, type, maxLen, value);
	
	if ( type == Chain(INPUT_TYPE_MENU))
	{
	    Chain menuValue;
	    getListValue(value, Chain(inputArray[i]), menuValue);
	    valList.Insert(menuValue);
	}
	else
	{
	    if ( inputArray[i][0] )
		valList.Insert(Chain(inputArray[i]).cutTrailing(" "));
	    else
		valList.Insert(Chain());
	}
    }
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return returnVal;
}

void Screen::setSelectedRow(int row)
{
    _rowSelected = row;
}

int Screen::getSelectedRow() const
{
    return _rowSelected;
}

int Screen::getVisibleRow() const
{
    return _vStart;
}

int Screen::showTableBox(const Chain& title, ListT<Chain>& schema,  const ListT<ListT<Chain> >& table,
			 const ListT<TableColor>& colorMap, int timeout,
			 bool isSelectable, int selectedRow, int visibleRow, int h )
{
	
    int ret = SC_TIMEOUT;
    
#ifndef LFCNOCURSES
    int c = 0;

    bool modFlag = false;
    bool showTable = true;

    _rowSelected = selectedRow;
    _vStart = visibleRow;

    if ( isSelectable == false || _rowSelected > table.Size() )
    {
	_rowSelected=1;
	_vStart=1;
    }
        
    int numCol=0;
    Chain *pA = schema.First();
    while ( pA )
    {
	Chain attr, attrLen;
	getKeyValue(*pA, attr, attrLen);
	numCol = numCol + attrLen.asInteger();	    
	pA = schema.Next();
    }

    WINDOW* tabwin = 0;

    int refreshCount=timeout;
	
    while ( showTable )
    {      	    	   	    

	int height;
	if ( h > 0 )
	    height = h;
	else
	    height=getHeight()-6;

	if ( tabwin == 0 )
	    tabwin = newwin(height + 3, numCol + 4, 2, 2);

	keypad(tabwin, TRUE);
	box(tabwin, 0, 0);
	
	switch(c)
	{	
	case KEY_UP:
	{
	    if ( isSelectable )
	    {
		if ( _rowSelected > 1 )
		{
		    if (_vStart >= _rowSelected && _vStart > 0)
			_vStart--;
		    _rowSelected--;
		}
	    }
	    break;
	}
	case KEY_DOWN:
	{
	    if ( isSelectable )
	    {
		if ( _rowSelected < table.Size() )
		{
		    if ( _rowSelected >= height + _vStart - 1 )
			_vStart++;		    
		    _rowSelected++;
		}
	    }
	    break;
	}
	case KEY_RETURN:
	{
	    modFlag=true;
	    showTable=false;
	    ret = SC_SELECT;
	    break;
	}
	case KEY_ESC:
	{
	    modFlag=true;
	    showTable=false;
	    ret = SC_LEAVE;
	    break;
	}
	case KEY_RESIZE:
	{
	    delwin(tabwin);
	    tabwin=0;
	    refreshFrame();
	    break;
	}
	default:
	{
	    ShortCut* pSC = _scList.First();
	    while ( pSC && modFlag == false)
	    {
		if ( c == pSC->getShortCut() )
		{
		    modFlag=true;
		    showTable=false;
		    ret = pSC->getCode();		    
		}
		pSC = _scList.Next();
	    }
	}
	}

	if ( tabwin != 0)
	{
	    int rowno=1;
	    int colno=2;
	    
	    wcolor_set(tabwin, COL_TITLE, NULL);	
	    wattron(tabwin, A_BOLD);
	    
	    mvwprintw(tabwin, 0, 2, "%s", (char*)title);
	    
	    pA = schema.First();
	    while ( pA )
	    {
		Chain attr, attrLen;
		getKeyValue(*pA, attr, attrLen);
		
		mvwprintw(tabwin, rowno, colno, "%s", (char*)attr);
		
		colno = colno + attrLen.asInteger();
		
		pA = schema.Next();
	    }
	    
	    rowno++;
	    
	    wattroff(tabwin, A_BOLD);
	    wcolor_set(tabwin, COL_CONTENT, NULL);
	    
	    int tabRow=_vStart;
	    int numRow = min(table.Size() - tabRow + 1, height );
	    
	    while ( tabRow < numRow + _vStart)
	    {		
		colno=2;
		
		int rowColor = COL_CONTENT;
		
		if ( _rowSelected == tabRow && isSelectable )
		{
		    wcolor_set(tabwin, COL_SELECTED, NULL);
		    wattron(tabwin, A_REVERSE);
		    rowColor = COL_SELECTED;
		}
		
		Chain *pA = schema.First();
		Chain *pVal = table[tabRow-1].First();
		while ( pA && pVal )
		{		       
		    TableColor *pTC = colorMap.Find(TableColor(*pVal));	       
		    if ( pTC )
		    {
			switch ( pTC->getCode() )
			{
			case SC_BLUE:
			    wcolor_set(tabwin, COL_BLUE, NULL);
			    break;
			case SC_GREEN:
			    wcolor_set(tabwin, COL_GREEN, NULL);
			    break;
			case SC_RED:
			    wcolor_set(tabwin, COL_RED, NULL);
			    break;
			case SC_YELLOW:
			    wcolor_set(tabwin, COL_YELLOW, NULL);
			    break;			
			}
		    }
		    
		    Chain attr, attrLen;
		    getKeyValue(*pA, attr, attrLen);
		    
		    for ( int i=0; i < attrLen.asInteger() ; i++)
			mvwprintw(tabwin, rowno, colno+1, "%s", " ");
		    mvwprintw(tabwin, rowno, colno, "%s", (char*)*pVal);
		    
		    colno = colno + attrLen.asInteger();
		    
		    wcolor_set(tabwin, rowColor, NULL);
		    
		    pVal = table[tabRow-1].Next();
		
		    pA = schema.Next();
		}
		
		if ( _rowSelected == tabRow && isSelectable )
		{		
		    wcolor_set(tabwin, COL_CONTENT, NULL);
		    wattroff(tabwin, A_REVERSE); 		
		}
		
		tabRow++;
		rowno++;
	    }
	    
	    wrefresh(tabwin);
	    
	    if ( modFlag == false )
	    {
		if ( timeout > 0 )
		{
		    bool go=true;
		    while ( refreshCount > 0 && go )
		    {
			wtimeout(tabwin, 1000);
			c = wgetch(tabwin);
			if ( c != -1 )
			    go=false;

			if ( go )
			{
			    int mainy, mainx;   
			    getmaxyx(stdscr, mainy, mainx);
			    mvwprintw(stdscr, mainy-1, ( refreshCount/1000 ) - 1, "%s", "*");
			
			    wrefresh(stdscr);
			    
			    refreshCount = refreshCount - 1000;
			}
		    }
		    if ( refreshCount == 0 )
		    {
			showTable=false;
			ret = SC_TIMEOUT;
		    }
		}
		else
		{
		    wtimeout(tabwin, MSEC_SLEEPDELAY); 	
		    c = wgetch(tabwin);
		}
	    }
	    else
	    {
		c = 0;
		modFlag = false;
	    }
	}
	else
	{
	    timeout(MSEC_SLEEPDELAY);
	    c = getch();			
	}
    }
    
    if ( tabwin )
	delwin(tabwin);
    
    refreshFrame();
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return ret;
}

int Screen::showAttributeBox(int keyLen, int valLen, const ListT< ListT<Chain> >& keyValueSpec, bool isVertical, int timeout)
{

    int ret = SC_TIMEOUT;
    
#ifndef LFCNOCURSES
    int c = 0;

    bool modFlag = false;
        
    bool showTable = true;

    int refreshCount=timeout;

    while ( showTable )
    {

	WINDOW* tabwin[10];
           	    	   	    
	switch(c)
	{	
	case KEY_RETURN:
	case KEY_ESC:
	{
	    modFlag=true;
	    showTable=false;
	    ret = SC_LEAVE;
	    break;
	}
	default:
	{
	    ShortCut* pSC = _scList.First();
	    while ( pSC && modFlag == false)
	    {
		if ( c == pSC->getShortCut() )
		{
		    modFlag=true;
		    showTable=false;
		    ret = pSC->getCode();		    
		}
		pSC = _scList.Next();
	    }
	}
	}

	ListT<Chain> *pKVL = keyValueSpec.First();

	int widx=0;
	int xpos=2;
	int ypos=2;
	
	while ( pKVL )
	{
	    tabwin[widx] = newwin(pKVL->Size() + 2, keyLen + valLen, ypos, xpos);
	    
	    if ( isVertical )
		ypos += pKVL->Size() + 2;
	    else
		xpos += keyLen +valLen + 2;
	    	    
	    noecho();		
	    keypad(tabwin[widx], TRUE);
	
	    box(tabwin[widx], 0, 0);
	
	    wcolor_set(tabwin[widx], COL_TITLE, NULL);
	
	    // wattron(tabwin, A_BOLD);

	    int rowno=1;
	    Chain *pKV = pKVL->First();
	    while ( pKV )
	    {
		Chain key, value;
		getKeyValue(*pKV, key, value);

		int colno=1;
		
		wcolor_set(tabwin[widx], COL_TITLE, NULL);
		mvwprintw(tabwin[widx], rowno, colno, "%s", (char*)key);

		colno += keyLen;
		
		wcolor_set(tabwin[widx], COL_CONTENT, NULL);
		mvwprintw(tabwin[widx], rowno, colno, "%s", (char*)value);
		
		rowno++;
		pKV = pKVL->Next();
	    }
	
	    wrefresh(tabwin[widx]);
	    widx++;
	    pKVL = keyValueSpec.Next();
	}
	    	
	if ( modFlag == false )
	{
	    if ( timeout > 0 )
	    {

		bool go=true;
		while ( refreshCount > 0 && go )
		{
		    wtimeout(tabwin[0], 1000);
		    c = wgetch(tabwin[0]);
		    if ( c != -1 )
			go=false;

		    if ( go )
		    {
			int mainy, mainx;   
			getmaxyx(stdscr, mainy, mainx);
			mvwprintw(stdscr, mainy-1, ( refreshCount/1000 ) - 1, "%s", "*");
			
			refresh();
			
			refreshCount = refreshCount - 1000;
		    }		
		}
		if ( refreshCount == 0 )		    
		    showTable=false;
	    }
	    else
	    {
		c = wgetch(tabwin[0]);
		if ( c == -1 )
		    showTable=false;
	    }
	}
	else
	{
	    c = 0;
	    modFlag = false;
	}

	// wtimeout(tabwin, timeout);
	// c = wgetch(tabwin);

	for ( int i=0; i<=widx; i++)
	    delwin(tabwin[widx]);

	refreshFrame();

    }

#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return ret;
}



void Screen::prevCursorPos(int attrListSize, int colOffset, int curRow, int curCol, int& prevCurRow, int& prevCurCol) 
{ 
    int okRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int okCol = OK_COL_POS; 

    int abortRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int abortCol = ABORT_COL_POS; 

    if ( curRow == CUR_ROW_OFFSET ) 
    { 
        prevCurRow = abortRow; 
        prevCurCol = abortCol; 
    } 
    else if ( curRow == okRow && curCol == okCol ) 
    { 
        prevCurRow = attrListSize + CUR_ROW_OFFSET - 1; 
        prevCurCol = colOffset; 
    } 
    else if ( curRow == abortRow && curCol == abortCol ) 
    { 
        prevCurRow = okRow; 
        prevCurCol = okCol; 
    } 
    else 
    { 
        prevCurRow = curRow-1; 
        prevCurCol = colOffset; 
    } 
} 

void Screen::nextCursorPos(int attrListSize, int colOffset, int curRow, int curCol, int& nextCurRow, int& nextCurCol, int navMode) 
{ 
    int okRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int okCol = OK_COL_POS; 

    int abortRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int abortCol = ABORT_COL_POS; 

    if ( curRow == attrListSize + CUR_ROW_OFFSET - 1) 
    { 
        nextCurRow = okRow; 
        nextCurCol = okCol; 
    } 
    else if ( curRow == okRow && curCol == okCol ) 
    { 
        nextCurRow = abortRow; 
        nextCurCol = abortCol; 
    } 
    else if ( curRow == abortRow && curCol == abortCol ) 
    { 
        nextCurRow = CUR_ROW_OFFSET; 
        nextCurCol = colOffset; 
    } 
    else if ( navMode == 0 )
    { 
        nextCurRow = curRow+1; 
        nextCurCol=colOffset; 
    }
    else
    {
        nextCurRow = okRow; 
        nextCurCol = okCol; 	
    }
} 

void Screen::getKeyValue(const Chain& s, Chain& key, Chain& value)
{
    Tokenizer t(s, _valSepToken);
    Chain v;
    t.nextToken(key);
    t.nextToken(value);
}

void Screen::getAttrTypeValue(const Chain& s, Chain& attr, Chain& type, int& maxLen, Chain& value)
{
    Tokenizer t(s, _attrSepToken);
    Chain v;
    t.nextToken(attr);
    t.nextToken(type);

    Chain maxLenStr;
    t.nextToken(maxLenStr);
    maxLen = maxLenStr.asInteger();
    
    t.nextToken(value);
}


void Screen::getListValue(const Chain& s, const Chain& key, Chain& value)
{    
    Tokenizer t1(s, _listSepToken);	
    Chain menuItem;
    while ( t1.nextToken(menuItem))
    {
	Tokenizer t2(menuItem, _valSepToken);
	Chain menuKey;
	while ( t2.nextToken(menuKey))
	{
	    if ( menuKey == key )
	    {
		t2.nextToken(value);
		return;
	    }
	}
    }
}

void Screen::getListKey(const Chain& s, const Chain& value, Chain& key)
{
    Tokenizer t1(s, _listSepToken);	
    Chain menuItem;
    while ( t1.nextToken(menuItem))
    {
	Tokenizer t2(menuItem, _valSepToken);
	Chain menuKey;
	Chain menuValue;
	
	if ( t2.nextToken(key) )
	{
	    if ( t2.nextToken(menuValue) )
	    {		
		if ( menuValue == value )
		{
		    return;
		}	
	    }
	}
    }
}

void Screen::formatMsg(const Chain& msg, Chain& formatedMsg, int width, int& height) 
{ 
   Tokenizer t(msg, Chain(" ")); 
   Chain tok; 

   int curLen=0; 
   height=1; 
   while ( t.nextToken(tok) ) 
   { 
      curLen += tok.length(); 
      if ( curLen > width ) 
      { 
          formatedMsg += Chain("\n"); 
          curLen=tok.length(); 
          height++; 
      } 
      formatedMsg += tok + Chain(" "); 
   } 
} 

void Screen::setStatusLine(const Chain& statusLine, int offset)
{
    _statusLine = statusLine;
    _statusOffset = offset;
    refreshFrame();
}

int Screen::getHeight()
{
    int height, width;   
    getmaxyx(stdscr, height, width);
    return height;
}

