#ifndef _SCREEN2_H_INCLUDED_ 
#define _SCREEN2_H_INCLUDED_ 
/////////////////////////////////////////////////////////////////////////////// 
// 
// Screen2.h 
// ---------
// Next generation Curses based screen interface definition 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2022 Bjoern Lemke 
// 
// INTERFACE MODULE 
// 
// Class: Screen 
// 
// Description: Description: NG Cursor based screen utiliy class
//
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 

#include <ctype.h> 
#include <string.h> 

#ifndef LFCNOCURSES

#ifdef HAVE_MINGW
#include <ncurses/ncurses.h>
#else
#include <ncurses.h>
#endif

#endif

#include "Chain.h" 
#include "ListT.h" 
#include "Datetime.h" 
#include "SigHandler.h" 

#define SC_RED 1
#define SC_GREEN 2
#define SC_YELLOW 3
#define SC_BLUE 4

#define ATTRSEPTOKEN ":"
#define LISTSEPTOKEN ","
#define VALSEPTOKEN "="

#define VAL_MAX_LEN 100 
#define VAL_MAX_NUM 20 

#define MAX_ATTR_WIN 5


class Screen2 : public SigHandler { 

public: 

    class Panel {
    public:
	Panel(Panel* pParent) { _pParent = pParent; _pNext = this; };
	virtual ~Panel() {};

	void setStatus(const Chain& status);

	// Chain& getStatus() { return _status; };

	virtual void show(bool doRec) = 0;
	virtual void handleKey(int c) = 0;
	
	virtual WINDOW* getWindow() = 0;

	virtual void enter() {};
	virtual void refresh() {};
	virtual bool doRefresh() { return false; };

	Panel* nextPanel() { return _pNext; };
	
    protected:

	Chain _status;
	Panel* _pParent;
	Panel* _pNext;
    };

    class Message : public Panel
    { 
    public: 
	Message(Panel* pParent);
	~Message();

	void setInfo(const Chain& title, const Chain& msg, int height = 0);
    
	void show(bool doRec);
	void handleKey(int c);
	WINDOW* getWindow();
	
    private:

	int _height;
	int _width;
	int _offset;
	int _msgHeight;
	

	Chain _title;
	Chain _msg;
	
	WINDOW *_msgwin;
    };

    class Select : public Panel { 
    public: 
	Select(Panel* pParent);
	~Select();

        void setItems(const Chain& title, const ListT<Chain>& items);

	// experimental

	/*
	void registerSelect( Panel* (*cb)(Select* pPanel, int selected) ) {
	    _cb = cb;
	}
	*/

        virtual Panel* onSelect(int selected) { return this; };

	Panel* setError(const Chain& title, const Chain& msg);
	
	void show(bool doRec);
        void handleKey(int c);
        WINDOW* getWindow();

	void setSelectedKey(const Chain& key);
	Chain getSelectedKey();
	Chain getSelectedValue();

    protected:
	
       Chain getItem(int i) { return _items[i]; };
	
    private:

	// experimentell
	// Panel* (*_cb)(Select* pPanel, int selected);
	
	Chain _title;
	int _height;
	int _width;
	int _highlight;
	ListT<Chain> _items;
	WINDOW *_selectwin;
	Message* _pError;
    };
   

    class GridColor { 
    public: 
	GridColor() {};
	GridColor(const Chain& val) {
	    _val = val;
	    _code = 0;
	}
	GridColor(const Chain& val, int code)  { 
	    _val = val; 
	    _code = code; 
	}; 
	~GridColor() {}; 
	const Chain& getValue() { return _val; }; 
	int getCode() { return _code; }; 
	GridColor& operator = (const GridColor& tc) { 
	    _val = tc._val; 
	    _code = tc._code; 
	    return(*this); 
	};
	
	bool operator == ( const GridColor& tc) const {
	    return _val == tc._val;
	};
	
    private: 
	Chain _val; 
	int _code; 
    };

    class Grid : public Panel { 
    public: 

	Grid(Panel *pParent);
	~Grid();
	
	void setSchema(const Chain& title, const ListT<Chain>& gridSchema, const ListT<GridColor>& colorMap, int height);
	
	// void setData( ListT< ListT<Chain> > &gridData);
	virtual Panel* onSelect(int rowNum) { return this; };
	virtual Panel* onKey(int c, int rowNum) { return this; };
	
	virtual ListT< ListT<Chain> > getData() {
	    ListT< ListT<Chain> > gridData;
	    return gridData;
	}

	Panel* setError(const Chain& title, const Chain& msg);
	
	void show(bool doRec);
	void handleKey(int c);
	WINDOW* getWindow();

    private:

	void refreshRow(int tabRow, int rowno);

	int _curStat;
	Chain _title;
	bool _isSelectable;
	int _initHeight;
	int _height;
	int _numCol;
	int _rowSelected;
	int _vStart;

	ListT<Chain> _schema;
	ListT< ListT<Chain> > _table;

	ListT<GridColor> _colorMap;
	
	WINDOW *_gridwin;
	Message* _pError;

    };
   
    
    class Confirm : public Panel { 
    public: 
	Confirm(Panel* pParemt);
	~Confirm();
	virtual Panel* onConfirm() { return _pParent; };
	virtual Panel* onCancel() { return _pParent; };

	void setInfo(const Chain& title, const Chain& details);

	Panel* setError(const Chain& title, const Chain& msg);
	
	
	void show(bool doRec);
	void handleKey(int c);
	WINDOW* getWindow();
	
    private:

	Chain _title;
	Chain _details;
    
	int _width;
	int _height;
    
	int _confVal;
	
	int _curRow; 
	int _curCol;

	Message* _pError;
	WINDOW* _confwin;
    };

    class Form : public Panel { 
    public: 
	Form(Panel *pParent);
	~Form();

	void setAttrList(const Chain& title, const ListT<Chain>& attrList, int maxVisible);
	void setValueList(const ListT<Chain>& valList);

	void clear();
	void reset();
	
	Panel* setError(const Chain& title, const Chain& msg);
	
	void show(bool doRec);
	void handleKey(int c);
	WINDOW* getWindow();
	
	virtual Panel* onConfirm(const ListT<Chain>& valList) { return _pParent; };
	virtual Panel* onCancel() { return _pParent; };
	
    private:
	
	Chain _title;
	ListT<Chain> _attrList;
	
	char _inputArray[VAL_MAX_NUM][VAL_MAX_LEN];
	
	int _maxLenArray[VAL_MAX_NUM];
	int _vposArray[VAL_MAX_NUM];
	
	int _colOffset;
	int _height;
	int _width;
	int _maxVisible;
	
	int _curRow;
	int _curCol;

	Message* _pError;
	WINDOW* _formwin;
	
	void getAttrTypeValue(const Chain& s, Chain& attr, Chain& type, int& maxLen, Chain& value);
	
	void getListKey(const Chain& s, const Chain& value, Chain& key);
	void getListValue(const Chain& s, const Chain& key, Chain& value);
	
	void prevCursorPos(int attrListSize, int colOffset, int curRow, int curCol,
			   int& prevCurRow, int& prevCurCol);
	
	void nextCursorPos(int attrListSize, int colOffset, int curRow, int curCol,
			   int& nextCurRow, int& nextCurCol, int navMode); 


	ListT<Chain> getValues();
	
	class FormSelect : public Select {
	    
	public:
	    
	    FormSelect(Panel* pParent) : Select(pParent) {		
	    }
	    
	    void setup(const Chain& title, const ListT<Chain>& items, char* pInput) {		
		_pInput = pInput;
		setItems(title, items);

		// setStatus(Chain("Setup select with ") + Chain(pInput));
		setSelectedKey(Chain(pInput));
	    }
		   
	    Panel* onSelect(int selected);
	    
	private:
	    char* _pInput;
	};

	FormSelect* _pSelect;

    };
    
    
    class Attribute : public Panel { 
    public:

	enum Layout { HORIZONTAL, VERTICAL, OVERLAY };
	
	Attribute(Panel* pParent, Layout layout);
	~Attribute();

	void setAttrList(const ListT<Chain>& titleList, const ListT< ListT<Chain> >& attrList);
	    
	void show(bool doRec);
	void handleKey(int c);
	WINDOW* getWindow();
	
    private:

	WINDOW* createWindow(int xpos, int ypos, const Chain& title, ListT<Chain> *pAttrList);
	Layout _layout;
	ListT<Chain> _titleList;
	int _visibleIdx;
	ListT< ListT<Chain> > _attrList;
	WINDOW* _attrwin[MAX_ATTR_WIN];
    };
    
    
    class Menu : public Panel
    { 
    public: 
	Menu();
	~Menu();
	
	void show(bool doRec);
	void handleKey(int c);
	WINDOW* getWindow();
	void regItem(const Chain& name, Panel *pPanel);
	
    private:

	
	class MenuItem { 
	public: 
	    MenuItem() { }; 
	    MenuItem(const Chain& name, Panel *pPanel) 
		{ 
		    _name = name; 
		    _pPanel = pPanel; 
		}; 
	    ~MenuItem() {}; 
	    const Chain& getName() { return _name; }; 
	    Panel* getPanel() { return _pPanel; }; 
	    MenuItem& operator = (const MenuItem& mi) 
		{ 
		    _name = mi._name; 
		    _pPanel = mi._pPanel; 
		    return(*this); 
		}; 
	private: 
	    Chain _name; 
	    Panel *_pPanel; 
	}; 

	ListT<MenuItem> _menu;
	int _menuSelected;
    };

  
    Screen2(); 
    ~Screen2(); 

    void setTimeout(int timeout);
    void regShortCut(char c, int code);
    void setRoot(Panel* pPanel);
    
    void showScreen();


    
private:
    

    class ShortCut { 
    public: 
        ShortCut() {}; 
        ShortCut(char c, int code) 
        { 
           _c = c; 
           _code = code; 
        }; 
        ~ShortCut() {}; 
        char getShortCut() { return _c; }; 
        int getCode() { return _code; }; 
        ShortCut& operator = (const ShortCut& sc) 
        { 
           _c = sc._c; 
           _code = sc._code; 
           return(*this); 
        }; 
    private: 
        char _c; 
        int _code; 
    };

    ListT<ShortCut> _scList; 

    void sigCatch(int sig); 

    Panel *_pPanel;

    int _timeout;
    int _statusOffset;

}; 


#endif 
